<?php
/*
 * star :: Simple Templates And Renderer.
 *
 * Star is a PHP HTML template engine in a single file.  It has no dependencies,
 * does not require Composer or ten thousand other files.  You can include it in
 * your code or copy/paste it onto the end of your script.  It supports
 * variables, inheritance, includes, and conditionals.  Templates are compiled
 * to PHP and cached at runtime for optimal performance.
 *
 * Version 1.2.0  November 17, 2025
 * Copyright (c) 2025, Ron Guerin <ron@vnetworx.net>
 *
 * Requires: PHP_PCRE, PHP 7.3 or later
 *
 * star is Free Software; you can redistribute it and/or modify it
 * (though only God knows why you'd want to) it under the terms of the
 * GNU General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * star is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * If you are not able to view the file COPYING, please write to the
 * Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * to get a copy of the GNU General Public License or to report a
 * possible license violation.
 *
 * @package star
 * @author Ron Guerin <ron@vnetworx.net>
 * @license http://www.fsf.org/licenses/gpl.html GNU Public License
 * @copyright Copyright &copy; 2025 Ron Guerin
 * @filesource
 * @link http://gothamcode.com/star
 * @version 1.2.0
 *
*/

define('VERSION_STAR', '1.2.0');
error_reporting(E_ALL);
ini_set('display_errors', 1);

function star_e($s) {
	return htmlspecialchars((string)$s, ENT_QUOTES | ENT_SUBSTITUTE | ENT_HTML5);
}

function star_esc_url($s) {
	return htmlspecialchars((string)$s, ENT_QUOTES | ENT_SUBSTITUTE | ENT_HTML5);
}

function star_raw_marker($s) {
	return ['__raw' => TRUE, 'html' => (string)$s];
}

function star_is_raw($v) {
	return is_array($v) && isset($v['__raw']) && $v['__raw'] === TRUE;
}

function star_ctx_get($ctx, $path) {
	if (($path === '') || ($path === NULL)) return NULL;
	if (is_array($ctx) && array_key_exists($path, $ctx)) return $ctx[$path];
	$parts = explode('.', $path);
	$cur = $ctx;
	foreach ($parts as $p) {
		if (is_array($cur) && array_key_exists($p, $cur)) $cur = $cur[$p];
		else return NULL;
	}
	return $cur;
}

function star_locate_template($file, $templatesbase) {
	$base = rtrim($templatesbase, '/\\');
	if (((($file !== '') && (DIRECTORY_SEPARATOR === '/')) ? ($file[0] === '/') : FALSE)) {
		if (file_exists($file)) return $file;
		if (file_exists($file.'.tpl')) return $file.'.tpl';
		else {
			echo 'Template not found: '.$file."\n";
			return FALSE;
		}
	}
	$full = $base.'/'.ltrim($file, '/\\');
	if (file_exists($full)) return $full;
	if (file_exists($full.'.tpl')) return $full.'.tpl';
	else {
		echo 'Template not found: '.$full."\n";
		return FALSE;
	}
}

function star_get_template_mtime($startFile, $templatesbase) {
	$maxmtime = 0;
	$chain = array();
	$cur = $startFile;
	while (TRUE) {
		$full = star_locate_template($cur, $templatesbase);
		if ($full === FALSE) return 0;
		$mtime = filemtime($full);
		if ($mtime > $maxmtime) $maxmtime = $mtime;
		$src = file_get_contents($full);
		$chain[] = $full;
		if (preg_match(chr(7).'\{\%\s*extends\s+[\'"]([^\'"]+)[\'"]\s*\%\}'.chr(7), $src, $m)) {
			$cur = $m[1];
			continue;
		}
		break;
	}
	return $maxmtime;
}

function star_merge_inheritance($startFile, $templatesbase) {
	$chain = array();
	$cur = $startFile;
	while (TRUE) {
		$full = star_locate_template($cur, $templatesbase);
		$src = file_get_contents($full);
		$chain[] = ['full' => $full, 'src' => $src, 'rel' => $cur];
		if (preg_match(chr(7).'\{\%\s*extends\s+[\'"]([^\'"]+)[\'"]\s*\%\}'.chr(7), $src, $m)) {
			$cur = $m[1];
			continue;
		}
		break;
	}
	$chain = array_reverse($chain);
	$blocks = array();
	foreach ($chain as $entry) {
		preg_match_all(chr(7).'\{\%\s*block\s+([a-zA-Z0-9_]+)\s*\%\}(.*?)\{\%\s*endblock\s*\%\}'
			.chr(7).'s', $entry['src'], $matches, PREG_SET_ORDER);
		foreach ($matches as $b) {
			$name = $b[1];
			$content = $b[2];
			if (isset($blocks[$name])) $content = str_replace('{{ super() }}', $blocks[$name], $content);
			$blocks[$name] = $content;
		}
	}
	$topsrc = $chain[0]['src'];
	$topsrc = preg_replace(chr(7).'\{\%\s*extends\s+[\'"]([^\'"]+)[\'"]\s*\%\}'.chr(7), '', $topsrc);
	$merged = preg_replace_callback(chr(7).'\{\%\s*block\s+([a-zA-Z0-9_]+)\s*\%\}(.*?)\{\%\s*endblock\s*\%\}'.chr(7).'s',
		function ($m) use ($blocks) {
			$name = $m[1];
			return isset($blocks[$name]) ? $blocks[$name] : $m[2];
		}
		, $topsrc
	);
	return $merged;
}

function star_compile_set_value($expr) {
	$expr = trim($expr);
	// String literal
	if ((($expr[0] === '"') || ($expr[0] === "'")) && ($expr[strlen($expr)-1] === $expr[0])) {
		return var_export(substr($expr, 1, -1), TRUE);
	}
	// Boolean/null
	if (strtolower($expr) === 'true') return 'true';
	if (strtolower($expr) === 'false') return 'false';
	if (strtolower($expr) === 'null') return 'null';
	// Variable or expression - compile to PHP
	return '('.star_compile_expr_to_php($expr).')';
}

function star_compile_expr_to_php($expr) {
	$expr = trim($expr);
	// Protect string literals
	$strings = array();
	$expr = preg_replace_callback(chr(7).'(["\'])(?:\\\\.|(?!\1).)*\1'.chr(7), function($m) use (&$strings) {
		$placeholder = '___STR'.count($strings).'___';
		$strings[$placeholder] = $m[0];
		return $placeholder;
	}, $expr);

	// Replace variables with star_ctx_get calls
	$expr = preg_replace_callback(chr(7).'\b([a-zA-Z_][a-zA-Z0-9_]*(?:\.[a-zA-Z_][a-zA-Z0-9_]*)*)\b'.chr(7),
		function($m) use (&$strings) {
			$var = $m[1];
			if (strpos($var, 'STR') !== FALSE) return $m[0];
			if (in_array(strtoupper($var), ['TRUE', 'FALSE', 'NULL'])) return strtolower($var);
			return 'star_ctx_get($ctx, '.var_export($var, TRUE).')';
		}, $expr);

	// Restore strings
	foreach ($strings as $placeholder => $str) $expr = str_replace($placeholder, $str, $expr);

	return $expr;
}

function star_eval_condition($expr, $ctx) {
	$original = $expr;

	// Normalize operators - must protect longer operators first
	$expr = str_replace('!==', '\x00STRICT_NEQ\x00', $expr);
	$expr = str_replace('===', '\x00STRICT_EQ\x00', $expr);
	$expr = str_replace('!=', '\x00NEQ\x00', $expr);
	$expr = str_replace('==', '\x00EQ\x00', $expr);
	$expr = str_replace('\x00NEQ\x00', '!==', $expr);
	$expr = str_replace('\x00EQ\x00', '===', $expr);
	$expr = str_replace('\x00STRICT_NEQ\x00', '!==', $expr);
	$expr = str_replace('\x00STRICT_EQ\x00', '===', $expr);

	// First, protect string literals by replacing them with placeholders
	// Use a placeholder that won't match the variable regex
	$strings = array();
	$expr = preg_replace_callback(chr(7).'(["\'])(?:\\\\.|(?!\1).)*\1'.chr(7), function($m) use (&$strings) {
		$placeholder = '___STRLIT'.count($strings).'___';
		$strings[$placeholder] = $m[0];
		return $placeholder;
	}, $expr);

	// Now replace variable names (won't match inside protected strings)
	$expr2 = preg_replace_callback(chr(7).'\b([a-zA-Z_][a-zA-Z0-9_]*(?:\.[a-zA-Z_][a-zA-Z0-9_]*)*)\b'.chr(7),
		function($m) use ($ctx, $strings) {
			$varname = $m[1];
			// Don't replace our placeholders
			if (strpos($varname, 'STRLIT') !== FALSE) return $m[0];
			if (in_array(strtoupper($varname), ['TRUE', 'FALSE', 'NULL'])) return strtolower($varname);
			$val = star_ctx_get($ctx, $varname);
			return var_export($val, TRUE);
		}
		, $expr
	);

	// Restore string literals
	foreach ($strings as $placeholder => $originalstring) {
		$expr2 = str_replace($placeholder, $originalstring, $expr2);
	}

	try {
		$result = eval('return ('.$expr2.');');
		return $result;
	}
	catch (DivisionByZeroError $e) {
		return FALSE;
	}
	catch (Throwable $e) {
		return FALSE;
	}
}

function star_find_matching_endif($src, $startpos) {
	static $callcount = 0;
	$callcount++;
	$depth = 1;
	$pos = $startpos;
	while ($depth > 0 && $pos < strlen($src)) {
		// Look for next if or endif - must be complete tags with word boundaries
		if (preg_match(chr(7).'\{\%\s*(?:(if)\s+.+?|(endif)\s*)\%\}'.chr(7), $src, $m, PREG_OFFSET_CAPTURE, $pos)) {
			$tagpos = $m[0][1];
			if (! empty($m[1][0])) $depth++; // matched 'if' with a condition after it
			else {
				// Matched 'endif'
				$depth--;
				if ($depth === 0) {
					$result = $tagpos + strlen($m[0][0]);
					return $result;
				}
			}
			$pos = $tagpos + strlen($m[0][0]);
		}
		else break;
	}
	return FALSE;
}

function star_compile_to_php($src, $templatesbase, $cachedir) {
	// Normalize whitespace-trimming markers
	$src = preg_replace([chr(7).'\\{%-\s*'.chr(7), chr(7).'\s*-%\\}'.chr(7),
		chr(7).'\{\{-\s*'.chr(7), chr(7).'\s*-\}\}'.chr(7)], ['{%','%}','{{','}}'], $src);

	// Remove comments {# ... #}
	$src = preg_replace(chr(7).'\{#.*?#\}'.chr(7).'s', '', $src);

	// Phase 1: Process set statements
	while (preg_match(chr(7).'\{\%\s*set\s+([a-zA-Z0-9_]+(?:\s*,\s*[a-zA-Z0-9_]+)*)\s*\%\}'
		.'(.*?)\{\%\s*endset\s*\%\}'.chr(7).'s', $src)) {
		$src = preg_replace_callback(chr(7).'\{\%\s*set\s+([a-zA-Z0-9_]+(?:\s*,\s*[a-zA-Z0-9_]+)*)\s*\%\}'
			.'(.*?)\{\%\s*endset\s*\%\}'.chr(7).'s',
			function($m) {
				$vars = array_map('trim', explode(',', $m[1]));
				$body = $m[2];
				$code = '<?php ob_start(); ?>'.$body.'<?php $__val = ob_get_clean(); ';
				foreach ($vars as $v) {
					$code .= '$ctx['.var_export($v, TRUE).'] = $__val; ';
				}
				$code .= '?>';
				return $code;
			}, $src);
	}

	while (preg_match(chr(7).'\{\%\s*set\s+([a-zA-Z0-9_]+(?:\s*,\s*[a-zA-Z0-9_]+)*)\s*=\s*'
		.'((?:[^%]|\%(?!\}))*)\s*\%\}'.chr(7), $src)) {
		$src = preg_replace_callback(chr(7).'\{\%\s*set\s+([a-zA-Z0-9_]+(?:\s*,\s*[a-zA-Z0-9_]+)*)'
			.'\s*=\s*((?:[^%]|\%(?!\}))*)\s*\%\}'.chr(7),
			function($m) {
				$vars = array_map('trim', explode(',', $m[1]));
				$exprs = array_map('trim', explode(',', $m[2]));
				$code = '<?php ';
				foreach ($vars as $idx => $v) {
					if (isset($exprs[$idx])) {
						$code .= '$ctx['.var_export($v, TRUE).'] = '.star_compile_set_value($exprs[$idx]).'; ';
					}
					else $code .= '$ctx['.var_export($v, TRUE).'] = null; ';
				}
				$code .= '?>';
				return $code;
			}, $src);
	}

	// Phase 2: Process includes first
	while (preg_match(chr(7).'\{\%\s*include\s+[\'"]([^\'"]+)[\'"](?:\s+with\s+(.*?))?\s*\%\}'.chr(7), $src)) {
		$src = preg_replace_callback(chr(7).'\{\%\s*include\s+[\'"]([^\'"]+)[\'"](?:\s+with\s+(.*?))?\s*\%\}'.chr(7),
			function($m) use ($templatesbase, $cachedir) {
				$incpath = $m[1];
				$paramslist = array();
				if (! empty($m[2])) {
					preg_match_all(chr(7).'([a-zA-Z0-9_]+)\s*=\s*(?:\'([^\']*)\'|"([^"]*)"|([a-zA-Z0-9_.]+))'.chr(7),
						$m[2], $pairs, PREG_SET_ORDER);
					foreach ($pairs as $p) {
						$key = $p[1];
						if ($p[2] !== '') $paramslist[] = array('key' => $key, 'type' => 'literal', 'value' => $p[2]);
						elseif ($p[3] !== '') $paramslist[] = array('key' => $key, 'type' => 'literal', 'value' => $p[3]);
						else $paramslist[] = array('key' => $key, 'type' => 'var', 'value' => $p[4]);
					}
				}
				$incmerged = star_merge_inheritance($incpath, $templatesbase);
				$inccompiled = star_compile_to_php($incmerged, $templatesbase, $cachedir);
				$code = '<?php $__saved_ctx = $ctx; ';
				foreach ($paramslist as $p) {
					if ($p['type'] === 'literal') {
						$code .= '$ctx['.var_export($p['key'], TRUE).'] = '.var_export($p['value'], TRUE).'; ';
					}
					else {
						$code .= '$ctx['.var_export($p['key'], TRUE)
							.'] = star_ctx_get($__saved_ctx, '.var_export($p['value'], TRUE).'); ';
					}
				}
				$code .= '?>';
				$code .= $inccompiled;
				$code .= '<?php $ctx = $__saved_ctx; ?>';
				return $code;
			}
			, $src
		);
	}

	// Phase 3: Process for loops with scoped context
	while (preg_match(chr(7).'\{\%\s*for\s+([a-zA-Z0-9_]+)\s+in\s+([a-zA-Z0-9_.]+)\s*\%\}'.chr(7), $src)) {
		$src = preg_replace_callback(chr(7).
			'\{\%\s*for\s+([a-zA-Z0-9_]+)\s+in\s+([a-zA-Z0-9_.]+)\s*\%\}(.*?)\{\%\s*endfor\s*\%\}'.chr(7).'s',
			function($m) {
				$var = $m[1];
				$listpath = $m[2];
				$body = $m[3];
				$compiled = '<?php $__list = star_ctx_get($ctx, '.var_export($listpath, TRUE).'); ';
				$compiled .= 'if (is_array($__list)) { foreach ($__list as $__k => $__v) { ';
				$compiled .= '$__saved_ctx = $ctx; ';
				$compiled .= '$ctx['.var_export($var, TRUE).'] = $__v; ';
				$compiled .= '$ctx['.var_export($var.'_key', TRUE).'] = $__k; ?>';
				$compiled .= $body;
				$compiled .= '<?php $ctx = $__saved_ctx; } } ?>';
				return $compiled;
			}
			, $src
		);
	}

	// Phase 4: Process conditionals last (innermost first)
	$maxiterations = 100;
	$iteration = 0;

	while ($iteration++ < $maxiterations) {
		$ifblocks = array();
		$searchpos = 0;
		while (preg_match(chr(7).'\{\%\s*if\s+(.+?)\s*\%\}'.chr(7), $src, $m, PREG_OFFSET_CAPTURE, $searchpos)) {
			$ifstart = $m[0][1];
			$condition = trim($m[1][0]);
			$afterif = $ifstart + strlen($m[0][0]);
			$endifend = star_find_matching_endif($src, $afterif);
			if ($endifend === FALSE) {
				$searchpos = $ifstart + 1;
				continue;
			}
			$ifblocks[] = ['start' => $ifstart, 'afterif' => $afterif, 'end' => $endifend, 'condition' => $condition];
			$searchpos = $ifstart + 1;
		}
		if (empty($ifblocks)) break;
		$innermost = NULL;
		foreach ($ifblocks as $block) {
			$bodystart = $block['afterif'];
			// Find where {% endif %} starts (not ends)
			$endifpattern = chr(7).'\{\%\s*endif\s*\%\}'.chr(7);
			if (preg_match($endifpattern, $src, $em, PREG_OFFSET_CAPTURE, $bodystart)) {
				// Search backwards from block end to find the endif that matches
				$potentialendif = $block['end'] - strlen($em[0][0]);
				$body = substr($src, $bodystart, $potentialendif - $bodystart);
				if (! preg_match(chr(7).'\{\%\s*if\s+'.chr(7), $body)) {
					$innermost = $block;
					break;
				}
			}
		}
		if ($innermost === NULL) break;

		// Extract body between {% if %} and {% endif %}
		// innermost['end'] is already the position after {% endif %}, calculated by star_find_matching_endif
		// We need to find where {% endif %} starts
		$searchfrom = $innermost['end'] - 20; // Search back at most 20 chars for {% endif %}
		if ($searchfrom < $innermost['afterif']) $searchfrom = $innermost['afterif'];

		if (preg_match(chr(7).'\{\%\s*endif\s*\%\}'.chr(7), $src, $endifmatch, PREG_OFFSET_CAPTURE, $searchfrom)) {
			 $endifstart = $endifmatch[0][1];
		}
		else break; // shouldn't happen

		$blockbody = substr($src, $innermost['afterif'], $endifstart - $innermost['afterif']);

		// Parse for elif/else
		$parts = array();
		$current = '';
		$currenttype = 'if';
		$currentcond = $innermost['condition'];
		$pos = 0;
		while ($pos < strlen($blockbody)) {
			if (preg_match(chr(7).'\{\%\s*(elif|else)(?:\s+(.+?))?\s*\%\}'.chr(7),
				$blockbody, $tm, PREG_OFFSET_CAPTURE, $pos)) {
				$tag = $tm[1][0];
				$tagpos = $tm[0][1];
				$current .= substr($blockbody, $pos, $tagpos - $pos);
				$parts[] = ['type' => $currenttype, 'condition' => $currentcond, 'body' => $current];
				$current = '';
				$currenttype = $tag;
				$currentcond = ($tag === 'elif' && isset($tm[2][0])) ? trim($tm[2][0]) : '';
				$pos = $tagpos + strlen($tm[0][0]);
			}
			else {
				$current .= substr($blockbody, $pos);
				break;
			}
		}
		$parts[] = ['type' => $currenttype, 'condition' => $currentcond, 'body' => $current];
		$compiled = '';
		foreach ($parts as $i => $part) {
			if ($i === 0) {
				$compiled .= '<?php if (star_eval_condition('.var_export($part['condition'], TRUE).', $ctx)) { ?>';
			}
			elseif ($part['} elseif type'] === 'elif') {
				$compiled .= '<?php } elseif (star_eval_condition('.var_export($part['condition'], TRUE).', $ctx)) { ?>';
			}
			elseif ($part['type'] === 'else') $compiled .= '<?php } else { ?>';
			$compiled .= $part['body'];
		}
		$compiled .= '<?php } ?>';

		// Replace the entire if block with compiled version
		$src = substr_replace($src, $compiled, $innermost['start'], $innermost['end'] - $innermost['start']);
	}

	// raw() & esc_url() when used inside {{ ... }}
	$src = preg_replace_callback(chr(7).'\{\{\s*(raw|esc_url)\s*\(\s*([a-zA-Z0-9_.]+'
		.'|\"[^\"]*\"|\'[^\']*\')\s*\)\s*\}\}'.chr(7),
		function($m) {
			$fn = $m[1];
			$arg = $m[2];
			if (($arg !== '') && (($arg[0] === '"') || ($arg[0] === '\''))) $getval = var_export(substr($arg, 1, -1), TRUE);
			else $getval = 'star_ctx_get($ctx, '.var_export($arg, TRUE).')';

			if ($fn === 'raw') {
				return '<?php $__v = '.$getval.'; if ($__v !== NULL) { echo (star_is_raw($__v)) '
					.'? $__v[\'html\'] : (string)$__v; } ?>';
			}
			return '<?php $__v = '.$getval.'; if ($__v !== NULL) { echo star_esc_url($__v); } ?>';
		}
		, $src
	);

	// Raw variables {{! var }}
	$src = preg_replace_callback(chr(7).'\{\{!\s*([a-zA-Z0-9_.]+)\s*\}\}'.chr(7),
		function($m) {
			return '<?php $__v = star_ctx_get($ctx, '.var_export($m[1], TRUE).'); '
				.'if ($__v !== NULL) { echo (star_is_raw($__v)) ? $__v[\'html\'] : (string)$__v; } ?>';
		}
		, $src
	);

	// URL variables {{@ var }}
	$src = preg_replace_callback(chr(7).'\{\{@\s*([a-zA-Z0-9_.]+)\s*\}\}'.chr(7),
		function($m) {
			return '<?php $__v = star_ctx_get($ctx, '.var_export($m[1], TRUE).'); '
				.'if ($__v !== NULL) { echo star_esc_url($__v); } ?>';
		}
		, $src
	);

	// Variables {{ var }}
	$src = preg_replace_callback(chr(7).'\{\{\s*([a-zA-Z0-9_.]+)\s*\}\}'.chr(7),
		function($m) {
			return '<?php $__v = star_ctx_get($ctx, '.var_export($m[1], TRUE).'); '
				.'if ($__v !== NULL) { '
				.'if (is_array($__v) && isset($__v[\'__raw\'])) { echo $__v[\'html\']; } '
				.'elseif (is_scalar($__v)) { echo star_e($__v); } '
				.'else { echo star_e(json_encode($__v)); } '
				.'} ?>';
		}
		, $src
	);

	return $src;
}

function star_render_string_recursive($src, $ctx=array(), $templatesbase=NULL, $currentdir=NULL) {
	// normalize whitespace-trimming markers
	$src = preg_replace([chr(7).'\\{%-\s*'.chr(7), chr(7).'\s*-%\\}'.chr(7),
		chr(7).'\{\{-\s*'.chr(7), chr(7).'\s*-\}\}'.chr(7)], ['{%','%}','{{','}}'], $src);

	// Remove comments {# ... #}
	$src = preg_replace(chr(7).'\{#.*?#\}'.chr(7).'s', '', $src);

	// Process set statements
	while (preg_match(chr(7).'\{\%\s*set\s+([a-zA-Z0-9_]+(?:\s*,\s*[a-zA-Z0-9_]+)*)\s*\%\}'
		.'(.*?)\{\%\s*endset\s*\%\}'.chr(7).'s', $src)) {
		$src = preg_replace_callback(chr(7).'\{\%\s*set\s+([a-zA-Z0-9_]+(?:\s*,\s*[a-zA-Z0-9_]+)*)'
			.'\s*\%\}(.*?)\{\%\s*endset\s*\%\}'.chr(7).'s',
			function($m) use ($ctx, $templatesbase, $currentdir) {
				$vars = array_map('trim', explode(',', $m[1]));
				$body = $m[2];
				$rendered = star_render_string_recursive($body, $ctx, $templatesbase, $currentdir);
				foreach ($vars as $v) {
					$ctx[$v] = $rendered;
				}
				return '';
			}, $src);
	}

	while (preg_match(chr(7).'\{\%\s*set\s+([a-zA-Z0-9_]+(?:\s*,\s*[a-zA-Z0-9_]+)*)'
		.'\s*=\s*((?:[^%]|\%(?!\}))*)\s*\%\}'.chr(7), $src)) {
		$src = preg_replace_callback(chr(7).'\{\%\s*set\s+([a-zA-Z0-9_]+(?:\s*,\s*[a-zA-Z0-9_]+)*)'
			.'\s*=\s*((?:[^%]|\%(?!\}))*)\s*\%\}'.chr(7),
			function($m) use ($ctx) {
				$vars = array_map('trim', explode(',', $m[1]));
				$exprs = array_map('trim', explode(',', $m[2]));
				foreach ($vars as $idx => $v) {
					if (isset($exprs[$idx])) {
						$expr = $exprs[$idx];
						if ((($expr[0] === '"') || ($expr[0] === "'")) && ($expr[strlen($expr)-1] === $expr[0])) {
							$ctx[$v] = substr($expr, 1, -1);
						}
						elseif (preg_match(chr(7).'^[a-zA-Z_][a-zA-Z0-9_]*(?:\.[a-zA-Z_][a-zA-Z0-9_]*)*$'.chr(7), $expr)) {
							$ctx[$v] = star_ctx_get($ctx, $expr);
						}
						else $ctx[$v] = star_eval_condition($expr, $ctx);
					}
					else $ctx[$v] = NULL;
				}
				return '';
			}, $src);
	}

	// Conditionals - process with proper nesting support
	while (preg_match(chr(7).'\{\%\s*if\s+(.+?)\s*\%\}'.chr(7), $src, $m, PREG_OFFSET_CAPTURE)) {
		$ifstart = $m[0][1];
		$condition = trim($m[1][0]);
		$afterif = $ifstart + strlen($m[0][0]);

		// Find matching endif
		$endifend = star_find_matching_endif($src, $afterif);
		if ($endifend === FALSE) break;

		// Extract and parse the block
		$fullblock = substr($src, $ifstart, $endifend - $ifstart);
		$blockbody = substr($fullblock, strlen($m[0][0]));
		$blockbody = preg_replace(chr(7).'\{\%\s*endif\s*\%\}$'.chr(7), '', $blockbody);

		// Parse top-level elif/else
		$parts = array();
		$current = '';
		$depth = 0;
		$pos = 0;
		$currenttype = 'if';
		$currentcond = $condition;

		while ($pos < strlen($blockbody)) {
			if (preg_match(chr(7).'\{\%\s*(if|elif|else|endif)\s*(?:(.+?)\s*)?\%\}'.chr(7),
				$blockbody, $tm, PREG_OFFSET_CAPTURE, $pos)) {
				$tag = $tm[1][0];
				$tagpos = $tm[0][1];

				$current .= substr($blockbody, $pos, $tagpos - $pos);

				if ($tag === 'if') {
					$depth++;
					$current .= $tm[0][0];
					$pos = $tagpos + strlen($tm[0][0]);
				}
				elseif ($tag === 'endif') {
					$depth--;
					$current .= $tm[0][0];
					$pos = $tagpos + strlen($tm[0][0]);
				}
				elseif (($tag === 'elif' || $tag === 'else') && $depth === 0) {
					$parts[] = ['type' => $currenttype, 'condition' => $currentcond, 'body' => $current];
					$current = '';
					$currenttype = $tag;
					$currentcond = ($tag === 'elif' && isset($tm[2][0])) ? trim($tm[2][0]) : '';
					$pos = $tagpos + strlen($tm[0][0]);
				}
				else {
					$current .= $tm[0][0];
					$pos = $tagpos + strlen($tm[0][0]);
				}
			}
			else {
				$current .= substr($blockbody, $pos);
				break;
			}
		}

		$parts[] = ['type' => $currenttype, 'condition' => $currentcond, 'body' => $current];

		// Evaluate and render
		$result = '';
		foreach ($parts as $part) {
			if ($part['type'] === 'if' || $part['type'] === 'elif') {
				if (star_eval_condition($part['condition'], $ctx)) {
					$result = star_render_string_recursive($part['body'], $ctx, $templatesbase, $currentdir);
					break;
				}
			}
			elseif ($part['type'] === 'else') {
				$result = star_render_string_recursive($part['body'], $ctx, $templatesbase, $currentdir);
				break;
			}
		}

		$src = substr_replace($src, $result, $ifstart, $endifend - $ifstart);
	}

	// For loops
	while (preg_match(chr(7).'\{\%\s*for\s+([a-zA-Z0-9_]+)\s+in\s+([a-zA-Z0-9_.]+)\s*\%\}'.chr(7), $src)) {
		$src = preg_replace_callback(chr(7).'\{\%\s*for\s+([a-zA-Z0-9_]+)\s+in\s+([a-zA-Z0-9_.]+)'
			.'\s*\%\}(.*?)\{\%\s*endfor\s*\%\}'.chr(7).'s',
			function($m) use ($ctx, $templatesbase, $currentdir) {
				$var = $m[1];
				$listpath = $m[2];
				$body = $m[3];
				$list = star_ctx_get($ctx, $listpath);
				if (! is_array($list)) return '';
				$out = '';
				foreach ($list as $k => $v) {
					$loop_ctx = $ctx;
					$loop_ctx[$var] = $v;
					$loop_ctx[$var.'_key'] = $k;
					$out .= star_render_string_recursive($body, $loop_ctx, $templatesbase, $currentdir);
				}
				return $out;
			}, $src);
	}

	// Includes
	$src = preg_replace_callback(chr(7).'\{\%\s*include\s+[\'"]([^\'"]+)[\'"](?:\s+with\s+(.*?))?\s*\%\}'.chr(7),
		function($m) use ($ctx, $templatesbase, $currentdir) {
			$incpath = $m[1];
			$params = array();
			if (! empty($m[2])) {
				preg_match_all(chr(7).'([a-zA-Z0-9_]+)\s*=\s*(?:\'([^\']*)\'|"([^"]*)"|([a-zA-Z0-9_.]+))'.chr(7),
					$m[2], $pairs, PREG_SET_ORDER);
				foreach ($pairs as $p) {
					$key = $p[1];
					if ($p[2] !== '') $val = $p[2];
					elseif ($p[3] !== '') $val = $p[3];
					else $val = star_ctx_get($ctx, $p[4]);
					$params[$key] = $val;
				}
			}
			$incctx = array_merge($ctx, $params);
			$merged = star_merge_inheritance($incpath, $templatesbase);
			return star_render_string_recursive($merged, $incctx, $templatesbase, $currentdir);
		}
		, $src
	);

	// raw() & esc_url()
	$src = preg_replace_callback(chr(7).'\{\{\s*(raw|esc_url)\s*\(\s*([a-zA-Z0-9_.]+'
		.'|\"[^\"]*\"|\'[^\']*\')\s*\)\s*\}\}'.chr(7),
		function($m) use ($ctx) {
			$fn = $m[1];
			$arg = $m[2];
			if (($arg !== '') && (($arg[0] === '"') || ($arg[0] === "'"))) $v = substr($arg, 1, -1);
			else $v = star_ctx_get($ctx, $arg);
			if ($v === NULL) return '';
			if ($fn === 'raw') return (star_is_raw($v)) ? $v['html'] : (string)$v;
			return star_esc_url($v);
		}
		, $src
	);

	// Raw variables {{! var }}
	$src = preg_replace_callback(chr(7).'\{\{!\s*([a-zA-Z0-9_.]+)\s*\}\}'.chr(7),
		function($m) use ($ctx) {
			$v = star_ctx_get($ctx, $m[1]);
			if ($v === NULL) return '';
			return (star_is_raw($v)) ? $v['html'] : (string)$v;
		}
		, $src
	);

	// URL variables {{@ var }}
	$src = preg_replace_callback(chr(7).'\{\{@\s*([a-zA-Z0-9_.]+)\s*\}\}'.chr(7),
		function($m) use ($ctx) {
			$v = star_ctx_get($ctx, $m[1]);
			if ($v === NULL) return '';
			return star_esc_url($v);
		}
		, $src
	);

	// Variables {{ var }}
	$src = preg_replace_callback(chr(7).'\{\{\s*([a-zA-Z0-9_.]+)\s*\}\}'.chr(7),
		function($m) use ($ctx) {
			$v = star_ctx_get($ctx, $m[1]);
			if ($v === NULL) return '';
			if (is_array($v) && isset($v['__raw'])) return $v['html'];
			if (is_scalar($v)) return star_e($v);
			return star_e(json_encode($v));
		}
		, $src
	);

	$src = preg_replace(chr(7)."[\r\n]+[ \t\r\n]+".chr(7), "\n", $src);
	return $src;
}

function star_render_template($templatename, $ctx=array(), $templatesbase=NULL, $cachedir=NULL) {
	$templatesbase = $templatesbase ? rtrim($templatesbase,'/\\') : (__DIR__.'/templates');
	$cachedir = $cachedir ? rtrim($cachedir,'/\\') : (__DIR__.'/cache');

	if (! is_dir($cachedir))	@mkdir($cachedir, 0755, TRUE);

	$cachekey = md5($templatename);
	$cachepath = $cachedir.'/'.$cachekey.'.php';

	$usecache = FALSE;
	if (file_exists($cachepath)) {
		$cachemtime = filemtime($cachepath);
		$sourcemtime = star_get_template_mtime($templatename, $templatesbase);
		if ($cachemtime >= $sourcemtime) {
			$usecache = TRUE;
		}
	}

	$merged = star_merge_inheritance($templatename, $templatesbase);

	if ($usecache) {
		ob_start();
		include($cachepath);
		$output = ob_get_clean();
		return preg_replace(chr(7)."[\r\n]+[ \t\r\n]+".chr(7), "\n", $output);
	}
	else {
		$compiled = star_compile_to_php($merged, $templatesbase, $cachedir);
		@file_put_contents($cachepath, $compiled);
		ob_start();
		eval('?>'.$compiled);
		$output = ob_get_clean();
		return preg_replace(chr(7)."[\r\n]+[ \t\r\n]+".chr(7), "\n", $output);
	}
}
